import 'dart:convert';

import 'package:get/get.dart';
import 'package:viserpay/core/utils/my_strings.dart';
import 'package:viserpay/core/utils/util.dart';
import 'package:viserpay/data/model/authorization/authorization_response_model.dart';
import 'package:viserpay/data/model/faq/faq_response_model.dart';
import 'package:viserpay/data/model/global/response_model/response_model.dart';
import 'package:viserpay/data/model/global/user/user_model.dart';
import 'package:viserpay/data/model/reward/reward_response_model.dart';
import 'package:viserpay/data/repo/reward/reward_repo.dart';
import 'package:viserpay/view/components/snack_bar/show_custom_snackbar.dart';

class RewardController extends GetxController {
  RewardRepo rewardRepo;
  RewardController({required this.rewardRepo});

  bool isLoading = false;
  List<LevelReward> levelRewards = [];
  List<EarnedPointModel> earnedPoint = [];
  List<ExpiredPoint> expirePoint = [];
  List<Reward> sortedRewards = [];
  List<Reward> rewardList = [];
  List<UserReward> userRedeemReward = [];
  List<UserReward> userPoint = [];
  List<FaqModel> faqs = [];
  GlobalUser? user;

  void initialize() {
    isLoading = true;
    levelRewards = [];
    sortedRewards = [];
    userPoint = [];
    userRedeemReward = [];
    faqs = [];
    user = null;
    isLoading = true;
    update();
    getRewardData();
  }

  Future<void> getRewardData() async {
    try {
      ResponseModel responseModel = await rewardRepo.getRewardHistory();
      if (responseModel.statusCode == 200) {
        LevelRewardResponseDataModel model = LevelRewardResponseDataModel.fromJson(jsonDecode(responseModel.responseJson));
        if (model.status == "success") {
          user = model.data?.user;
          levelRewards = model.data?.levelRewards ?? [];
          faqs = model.data?.rewardFaq ?? [];
          userPoint = model.data?.usedPoints ?? [];
          userRedeemReward = model.data?.redeemRewards ?? [];
          earnedPoint = model.data?.earnedPoints ?? [];
          expirePoint = model.data?.expiredPoints ?? [];
          if (levelRewards.isNotEmpty) {
            sortRewards(levelRewards.first);
          }
          update();
        } else {
          CustomSnackBar.error(errorList: model.message?.error ?? [MyStrings.somethingWentWrong]);
        }
      } else {
        CustomSnackBar.error(errorList: [responseModel.message]);
      }
    } catch (e) {
      printX(e);
    } finally {
      isLoading = false;
      update();
    }
  }

  void sortRewards(LevelReward levelReward) {
    sortedRewards = levelReward.rewards ?? [];
    update();
  }

  LevelReward getLevelByUserPoints() {
    for (int i = 0; i < levelRewards.length; i++) {
      var element = levelRewards[i];
      double minPoints = double.parse(element.minPoints ?? "0");
      double maxPoints = double.parse(element.maxPoints ?? "0");
      double userPoints = double.parse(user?.totalPoints ?? "0");
      if (userPoints >= minPoints && userPoints <= maxPoints) {
        return levelRewards[i]; // Levels start from 1
      }
    }

    return LevelReward(name: "");
  }

  int selectedRewardIndex = 0;
  selectRewardIndex(int ind) {
    selectedRewardIndex = ind;
    rewardList = levelRewards[ind].rewards ?? [];
    update();
  }

  String selectedId = "-1";
  Future<void> redeemReward(Reward reward) async {
    selectedId = reward.id.toString();
    update();
    try {
      ResponseModel responseModel = await rewardRepo.redeemReward(reward.id.toString());
      if (responseModel.statusCode == 200) {
        AuthorizationResponseModel model = AuthorizationResponseModel.fromJson(jsonDecode(responseModel.responseJson));
        if (model.status == "success") {
          userRedeemReward.add(UserReward(reward: reward, rewardId: reward.id));
          getRewardData();
          CustomSnackBar.success(successList: model.message?.success ?? [MyStrings.succeed]);
        } else {
          CustomSnackBar.error(errorList: model.message?.error ?? [MyStrings.somethingWentWrong]);
        }
      } else {
        CustomSnackBar.error(errorList: [responseModel.message]);
      }
    } catch (e) {
      printX(e);
    } finally {
      selectedId = "-1";
      update();
    }
  }

  bool claimed(String rewardId) {
    return userRedeemReward.any((ele) => ele.rewardId == rewardId);
  }

  double getProgressValue(String maxPoint) {
    try {
      double maxPoints = double.parse(maxPoint);
      double userPoints = double.parse(user?.totalPoints ?? "0");
      return userPoints / maxPoints;
    } catch (e) {
      return 0.5;
    }
  }
}
